/**
 * @addtogroup ssalibrary
 * @{
 */

/**
 * @defgroup ssacapi The SSA VPN C API (CAPI)
 * @{
 *
 * The VPN C API interface to the standard VPN API. 
 * These funtions are wrapper functions that allow applications
 * to use standard C-like calls into the VPN API. It's main
 * purpsose is to avoid having to redistribute DDK libraries and
 * headers.
 *
 */
/**-----------------------------------------------------------------
 * @file    vpncapi.cpp
 * @date    Jan 2011
 * @brief   Pure VPN API
 *
 * Copyright (c) 2011 by Cisco Systems, Inc.
 * All rights reserved.
 *------------------------------------------------------------------
 */
#include "vpncapi.h"
#include "PrivateApi.h"
#ifdef _WIN32
#include "atlconv.h"
#endif

/** 
 * This is a private instance of the HVPNAPI. It instantiates 
 * a PrivateApi class which encapsulates the C++ and callback code.
 */
struct _HVPNAPI : PrivateApi
{
    /**
     * These are the callbacks that will be invoked when certain VPN events occur
     */
    _HVPNAPI(struct VPNAPI_CALLBACKS* pCallbacks): PrivateApi(pCallbacks)
	{

	}
};


/**
 * @name    ssa_vpnapi_create
 * @brief   Creates a CAPI VPN context handle
 * @ingroup ssalibrary
 *
 * This function will be used to create a CAPI VPN handle. This handle will
 * be used for all other CAPI function calls. 
 * 
 * @param [in] pCallbacks - This is a pointer to the user supplied callbacks
 *                          which will be called when specific VPN events
 *                          occur.
 *
 * @retval handle - This handle must be used for subsequent ssa_vpnapi
 *                  function calls.
 */
HVPNAPI* vpnapi_create(struct VPNAPI_CALLBACKS* pCallbacks)
{
    return new HVPNAPI(pCallbacks);
}

/**
 * @name    ssa_vpnapi_delete
 * @brief   Delete a CAPI VPN context handle
 * @ingroup ssalibrary
 *
 * This function will be used to delete a CAPI VPN handle. The handle must
 * have been previously open/created using ssa_vpnapi_create().
 * 
 * @param [in] pHandle - This is the handle provided from the
 *                       ssa_vpnapi_create() call.
 */
void vpnapi_delete(HVPNAPI* pHandle)
{
    delete pHandle;
    pHandle = NULL;
}

/**
 * @name   ssa_vpnapi_attach
 * @brief  Attach to the VPN agent/subsystem
 * @ingroup ssalibrary
 *
 * This API must be called prior to attempting connections,
 * retrieving statistics, etc.  If successful, this function returns
 * true.  If not successful, it returns false and returns a notice error
 * message to the user.
 *
 * A single call to this method is all that is necessary.  If the
 * attach fails, a message indicating the VPN service is not available
 * is returned.  If the call succeeds, the ServiceReadyCB is
 * called and true is returned.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] SBL - Start Before Logon is a mode of operation where a GUI
 * can be launched prior to the normal windows logon sequence.  This
 * allows a VPN tunnel to be activated and used as part of the windows
 * logon sequence.  The SBL attribute should be set to true only when
 * it has been passed as an argument to the program by the VPN agent.
 *
 * @param [in] isGUI - Indicates that the started program is a UI
 * application.  With this attribute set to true, the application will
 * now receive WMHints.
 *
 * @param [in] requestFullCapabilities - Indicates that the client program is
 * requesting full API capabilities.  Full capabilities allows the
 * client program to connect, disconnect, receive statistics, etc.
 * When full capabilities are not requested or not available, the
 * client program will not be able to establish new VPN connections.
 * Only a client program with full capabilites can do this.  In
 * addition, only the first program requesting full capabilities will
 * be granted this level of access.  The attach method can succeed
 * even if full capabilities is requested but not granted.  To test
 * for this state, use the method ::hasFullCapabilities.
 *
 * @param [in] suppressAutoConnect - Indicates that the client wishes to
 * override automatically initiating a connection to the last connected
 * secure gateway at startup.  Normally, this is determined by the 
 * value of the AutoConnectOnStart preference.  If this flag is true
 * then an automatic connection will never be initiated, even if 
 * AutoConnectOnStart is enabled.
 */
int vpnapi_attach(HVPNAPI* pHandle, 
                  int SBL, 
                  int isGUI, 
                  int requestFullCapabilities, 
                  int suppressAutoConnect)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }
    return pHandle->attach((SBL != 0),
                           (isGUI != 0),
                           (requestFullCapabilities != 0),
                           (suppressAutoConnect != 0));
}

/**
 * @name   ssa_vpnapi_process_events
 * @brief  Process VPN Events
 * @ingroup ssalibrary
 *
 * If a pEventAvailableCB is supplied during the ssa_vpnapi_create() 
 * function call, this function must be called to process events after
 * that callback is invoked.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 */
void vpnapi_process_events(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->ProcessEvents();
}


/**
 * @name   ssa_vpnapi_connect
 * @brief  Connect to a specified host
 * @ingroup ssalibrary
 *
 * This API initiates a connection to the specified host.
 * The connection results in the presentation of authentication
 * credentials, as appropriate.  Any credentials returned by the secure
 * gateway are delivered via the pUserPromptCB callback.
 *
 * See ConnectPromptInfo for more details on possible authentication
 * credentials.
 *
 * If the connection request is accepted, true is returned.  This does
 * not mean the connection succeeded.  If the connection succeeds, a
 * state of connect will be received via the pStateCB callback.
 * 
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] host - The host name or ip address to connect to (secure gateway).
 *
 * @retval true - The connection request was accepted. This does not mean that
 * the connection to the secure gateway has been established.
 *
 * @retval false - The connection request was refused. This could mean that an
 * attech was not done prior to the connect. A notice callback will be
 * generated with the failure message.
 */
int vpnapi_connect(HVPNAPI* pHandle, const char* host)
{
    USES_CONVERSION;
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }
    return pHandle->connect(A2T(host));
}

/**
 * @name   ssa_vpnapi_set_new_tunnel_group
 * @brief  Set the actively selected group
 * @ingroup ssalibrary
 *
 * Use this API to change selected group after initial connection
 * request has been made and credentials were delivered. 
 *
 * Depending on secure gateway configuratiion, call to this API may
 * result in a new connection request and will update credentials 
 * required for the selected group. New credentials returned by the 
 * secure gateway are delivered via the pUserPromptCB callback.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] group - This is the new group being requested.
 *
 */
int vpnapi_set_new_tunnel_group(HVPNAPI* pHandle, const char* group)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }

    USES_CONVERSION;
    return pHandle->setNewTunnelGroup(A2T(group));
}

/**
 * @name   ssa_vpnapi_user_submit
 * @brief  Set the actively selected group
 * @ingroup ssalibrary
 *
 * Call this API to indicate that authentication credential
 * requests values solicited by the pUserPromptCB method can now
 * be read from the ConnectPromptInfo instance.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 */
void vpnapi_user_submit(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    return pHandle->UserSubmit();
}

/**
 * @name   ssa_vpnapi_set_banner_response
 * @brief  Set the actively selected group
 * @ingroup ssalibrary
 *
 * Call this method after a pBannerCB has been received to indicate 
 * that the user response to the banner can now be read.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] bAccept - Indicates if the user accepted or declined the banner.
 *
 */
void vpnapi_set_banner_response(HVPNAPI* pHandle, int bAccept)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->setBannerResponse((bAccept != 0));
}


/**
 * @name   ssa_vpnapi_prompt_set_entry_value
 * @brief  Set the value for a specific prompt
 * @ingroup ssalibrary
 *
 * Call this API to set the user selection.  If a default value is
 * present, it's value will be used unless this method in invoked.
 * Returns true if the value is successfully set.
 * ssa_vpnapi_user_submit() should be called to submit the prompt values.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] pszEntryName - This is the prompt name for which a value will be set.
 * 
 * @param [in] pszEntryValue - This is the value to set for a specificed
 * prompt name.
 *
 * @retval true - The value was successfully set
 * @retval false - The value was not successfully set
 */
int vpnapi_prompt_set_entry_value(HVPNAPI* pHandle, 
                                  const char* pszEntryName, 
                                  const char* pszEntryValue)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }
    return pHandle->CSetPromptEntryValue(pszEntryName, pszEntryValue);
}

/**
 * @name   ssa_vpnapi_prompt_set_enrollment_ca
 * @brief  Setup to retrieve a certificate from the CA
 * @ingroup ssalibrary
 *
 * Call this API to indicate that we want to retrieve a certificate from the CA.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] bEnroll - True to indicate that we want to retrieve a
 * certificate and false to indicate that we do not.
 */
void vpnapi_prompt_set_enrollment_ca(HVPNAPI* pHandle, int bEnroll)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    return pHandle->setEnrollmentCA((bEnroll != 0));
}

/**
 * @name   vpnapi_request_importpkcs12
 * @brief  Request the import of a PKCS12 Certificate
 * @ingroup ssalibrary
 *
 * Call this function to request that the provided data be imported as a 
 * PKCS12 certifcate.
 *
 * @param [in] pHandle - This is the handle provided from the
 * vpnapi_create() call.
 *
 * @param [in] pPKCS12Data - Pointer to a buffer containing the bytes to
 * be interpreted as a PKCS12 file.
 *
 * @param [in] uiDataLen - The length of the buffer indicated by pPKCS12Data.
 *
 * @retval true - The import request has been acknowledged and is being
 * processed.
 * @retval false - The import request has failed or been denied.
 */
int vpnapi_request_importpkcs12(HVPNAPI* pHandle,
                                unsigned char *pPKCS12Data,
                                unsigned int uiDataLen)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return 0;
    }

    if (NULL == pPKCS12Data || uiDataLen == 0)
    {
        vpncapi_log(LOGERR, "Input indicated a bad import buffer");
        return 0;
    }

    std::vector<unsigned char> importData(uiDataLen, 0);
    memcpy(&(importData[0]), pPKCS12Data, uiDataLen);
    return (pHandle->requestImportPKCS12(importData)) ? 1 : 0;
}

/**
 * @name   vpnapi_request_import_localization
 * @brief  Request the import of a gettext localization .mo file
 * @ingroup ssalibrary
 *
 * Call this function to request that the provided data be imported as a 
 * PKCS12 certifcate.
 *
 * @param [in] pHandle - This is the handle provided from the
 * vpnapi_create() call.
 *
 * @param [in] pLocaleData - Pointer to a buffer containing the bytes to
 * be interpreted as a gettext .mo localization file.
 *
 * @param [in] uiDataLen - The length of the buffer indicated by pLocaleData.
 *
 * @retval true - The import request succeeded.
 * @retval false - The import request failed.
 */
int vpnapi_request_import_localization(HVPNAPI* pHandle,
                                       const char *pLocale,
                                       unsigned char *pLocaleData,
                                       unsigned int uiDataLen)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return 0;
    }

    if (NULL == pLocale)
    {
        vpncapi_log(LOGERR, "Locale was not indicated--must be specified.");
        return 0;
    }

    if (NULL == pLocaleData || uiDataLen == 0)
    {
        vpncapi_log(LOGERR, "Input indicated a bad import buffer");
        return 0;
    }

    USES_CONVERSION;
    tstring tstrLocale(A2T(pLocale));
    std::vector<unsigned char> importData(uiDataLen, 0);
    memcpy(&(importData[0]), pLocaleData, uiDataLen);

    return (pHandle->requestImportLocalization(tstrLocale, importData)) ? 1 : 0;
}

/**
 * @name   vpnapi_disconnect
 * @brief  Disconnect a VPN tunnel connection
 * @ingroup ssalibrary
 *
 * Call this API to disconnect the current VPN session from the secure gateway.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 */
void vpnapi_disconnect(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->disconnect();
}

/**
 * @name   ssa_vpnapi_detach
 * @brief  Detach from the VPN agent/subsystem
 * @ingroup ssalibrary
 *
 * Call this API to detach from the VPN subsystem. This is the opposite to
 * ssa_vpnapi_attach().
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 */
void vpnapi_detach(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->detach();
}

/**
 * @name   ssa_vpnapi_get_preferences
 * @brief  Get the saved preferences
 * @ingroup ssalibrary
 *
 * Call this API to get all user controllable preferences. The reference 
 * returned is valid until ssa_vpnapi_save_preferences() is called.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @retval NULL - Failure occurred, handle invalid.
 * @retval PREFERENCE_INFO - A reference of the PREFERENCE_INFO structure is
 * returned.
 */
const struct PREFERENCE_INFO* vpnapi_get_preferences(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return NULL;
    }
    return pHandle->CGetPreferences();
}

/**
 * @name   ssa_vpnapi_save_preferences
 * @brief  Save the preferences which were previously set.
 * @ingroup ssalibrary
 *
 * Call this API to save the currently set preferences.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @retval 0/false - Failure occurred, handle invalid.
 * @retval true - Preferences were saved successfully.
 */
int vpnapi_save_preferences(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return 0;
    }
    return pHandle->CSavePreferences();
}

/**
 * @name   ssa_vpnapi_preference_set_value
 * @brief  Set the specified preference
 * @ingroup ssalibrary
 *
 * Call this API to set a specified preference
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 * 
 * @param [in] pszPreferenceName -
 *
 * @param [in] pszValue -
 * 
 * @retval 0/false - Failure occurred, handle invalid.
 * @retval true - The value was set for the specified preference.
 *
 */
int vpnapi_preference_set_value(HVPNAPI* pHandle, const char *pszPreferenceName, const char* pszValue)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return 0;
    }
    return pHandle->CPreferenceSetValue(pszPreferenceName, pszValue);
}

/**
 * @name   ssa_vpnapi_get_preference_by_name
 * @brief  Returns the requested named preference
 * @ingroup ssalibrary
 *
 * Call this API to returned the requested preference.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] pszPreferenceName - The name of the preference to return a
 * value.
 *
 * @retval NULL - Failure, argument was invalid
 *
 * @retval Else - Reference to a PREFERENCE structure containing the value(s)
 * of the requested preference name.
 */
const struct PREFERENCE* vpnapi_get_preference_by_name(HVPNAPI* pHandle, const char* pszPreferenceName)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return NULL;
    }
    return pHandle->CGetPreferenceByName(pszPreferenceName);
}

/**
 * @name   ssa_vpnapi_get_stats
 * @brief  Get the VPN statistics
 * @ingroup ssalibrary
 *
 * Call this API to force the statistics to be updated and the
 * vpnapi_stats_cb() will be called.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 */
void vpnapi_get_stats(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->getStats();
}

/**
 * @name   ssa_vpnapi_get_state
 * @brief  Get the current VPN state
 * @ingroup ssalibrary
 *
 * Call this API to have VPN update the current state and call the
 * ssa_vpnapi_state_cb() callback function.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 */
void vpnapi_get_state(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->getState();
}

/**
 * @name   ssa_vpnapi_reset_stats
 * @brief  Reset the VPN statistics
 * @ingroup ssalibrary
 *
 * Call this API the reset the current statistics back to their initial state.
 * This will cause the ssa_vpnapi_stats_cb() callback to be invoked.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 */
void vpnapi_reset_stats(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->resetStats();
}

/**
 * @name   ssa_vpnapi_is_connected
 * @brief  Determine if there is a VPN connection
 * @ingroup ssalibrary
 *
 * Call this API to determine if there is a VPN connection established
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @retval true - VPN connection is established
 * @retval false - No VPN connection is established
 */
int vpnapi_is_connected(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }
    return pHandle->isConnected();
}

/**
 * @name   ssa_vpnapi_is_available
 * @brief  Determine is the VPN subsystem can be used
 * @ingroup ssalibrary
 *
 * Call this API to determine if the VPN subsystem is enabled and useable.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 * 
 * @retval true - The VPN subsystem is enabled and useable.
 * @retval false - The VPN subsystem is disable and not useable.
 */
int vpnapi_is_available(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }
    return pHandle->isAvailable();
}

/**
 * @name   ssa_vpnapi_vpn_service_available
 * @brief  Determine is the VPN service is available.
 * @ingroup ssalibrary
 *
 * Call this API to determine is the VPN subsystem is ready to be attached.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 * 
 * @retval true - VPN service is available for an attach.
 * @retval false - VPN service is not available for an attach.
 */
int vpnapi_is_vpn_service_available(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }
    return pHandle->isVPNServiceAvailable();
}

/**
 * @name   ssa_vpnapi_operating_mode
 * @brief  Determine the operating mode
 * @ingroup ssalibrary
 *
 * Call this API to determine if the current mode of the VPN subsystem
 * matchs a specified mode.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @param [in] mode - Compare this mode to the current operating mode.
 *
 * @retval true - The specified operating mode is equal to the current VPN
 * operating mode.
 * @retval false - The specified operating mode does not match the current VPN
 * operating mode.
 */
int vpnapi_is_operating_mode(HVPNAPI* pHandle, enum OperatingMode mode)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return false;
    }
    return pHandle->isOperatingMode(mode);
}

/**
 * @name   ssa_vpnapi_get_default_hostname
 * @brief  Get the default hostname
 * @ingroup ssalibrary
 *
 * Call this API to get the default hostname being used
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @retval NULL - Invalid argument
 * @retval Else - Pointer to the default hostname string.
 */
const char* vpnapi_get_default_hostname(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return NULL;
    }
    return pHandle->CGetDefaultHostname();
}

/**
 * @name   ssa_vpnapi_get_hostnames
 * @brief  Get the hostnames
 * @ingroup ssalibrary
 *
 * Call this API to get the list of hostnames
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @retval empty - A NULL string list indicates a failure
 * @retval STRING_LIST - A list of hostname strings.
 */
const struct STRING_LIST vpnapi_get_hostnames(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        STRING_LIST empty = {NULL, 0};
        return empty;
    }
    return pHandle->CGetHostnames();
}

/**
 * @name   ssa_vpnapi_is_event_available
 * @brief  
 * @ingroup ssalibrary
 *
 * Call this API in order to poll for events.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 *
 * @retval false - Invalid handle or no event is available
 * @retval true - Event is available for processing
 */
int vpnapi_is_event_available(HVPNAPI* pHandle)
{
	if (NULL == pHandle)
	{
        vpncapi_log(LOGERR, "vpncapi handle NULL");
		return 0;
	}
    return pHandle->CIsEventAvailable();
}

/**
 * @name   ssa_vpnapi_prompt_get_entry_names
 * @brief  
 * @ingroup ssalibrary
 *
 * Call this API to retrieve the string names of all the prompt entries.
 *
 * @param [in] pHandle - This is the handle provided from the
 * ssa_vpnapi_create() call.
 */
const struct STRING_LIST vpnapi_prompt_get_entry_names(HVPNAPI* pHandle)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        STRING_LIST empty = {NULL, 0};
        return empty;
    }
    return pHandle->CGetPromptEntryNames();
}

/**
 * @name   ssa_vpnapi_get_prompt_entry
 * @brief  Retrieve a particular prompt_entry by string name
 * @ingroup ssalibrary
 *
 * Call this API function in order to retrieve the desired prompt entry.
 *
 * @param [in] pPrompt - The CONNECT_PROMPT_ENTRY provided as a argument to the
                         user prompt callback.
 * @param [in] pPromptName - The particular element you are looking to retrieve
 *
 * @retval NULL - Prompt specified was not found in the list
 * @retval PROMPT_ENTRY - Returns a pointer to the PROMPT_ENTRY found matching
 * the pPromptName specified.
 */
const struct PROMPT_ENTRY *vpnapi_get_prompt_entry(const struct CONNECT_PROMPT_INFO *pPrompt, const char *pPromptName)
{
    if (pPrompt == NULL || pPromptName == NULL)
    {
        vpncapi_log(LOGERR, "vpncapi_get_prompt_entry: Bad(NULL) parameter passed");
        return NULL;
    }
    for (unsigned int i = 0; i < pPrompt->nEntries; i++)
    {
        const struct PROMPT_ENTRY* pEntry = pPrompt->ppEntries[i];

        if ( strcmp(pEntry->pszName, pPromptName) == 0 )
        {
            return(pEntry);
        }
    }
    return NULL;
}

/**
TODO
*/
void vpnapi_prompt_set_canceled(HVPNAPI* pHandle, int isCanceled)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->setCanceled((isCanceled > 0) ? true : false );
}

/**
 * @name   vpnapi_set_cert_blocked_response
 * @brief  indicates the user's response to the server certificate blocked error
 * @ingroup ssalibrary
 *
 * Call this method after a vpnapi_cert_blocked_cb has been received to
 * indicate the user's response to the blocked untrusted VPN server
 * error message.
 *
 * @param [in] pHandle - This is the handle provided from the
 * vpnapi_create() call.
 *
 * @param [in] bUnblock - This indicates whether user wishes to unblock
 * untrusted server connections
 *
 */
void vpnapi_set_cert_blocked_response(HVPNAPI* pHandle, int bUnblock)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->setCertBlockedResponse((bUnblock > 0) ? true : false);
}

/**
 * @name   vpnapi_set_cert_warning_response
 * @brief  indicates the user's response to the server certificate warning
 * @ingroup ssalibrary
 *
 * Call this method after a vpnapi_cert_warning_cb has been received to
 * indicate the user's response to the server certificate error
 * warning
 *
 * @param [in] pHandle - This is the handle provided from the
 * vpnapi_create() call.
 *
 * @param [in] bConnect - This indicates whether user wishes to connect to
 * the untrusted server
 *
 * @param [in] bImportCert - This indicates whether user wishes to import the server
 * certificate so that they are not warned again for the same server connection
 */
void vpnapi_set_cert_warning_response(HVPNAPI* pHandle, int bConnect, int bImportCert)
{
    if (NULL == pHandle)
    {
        vpncapi_log(LOGERR, "vpncapi handle NULL");
        return;
    }
    pHandle->setCertWarningResponse((bConnect > 0) ? true : false,
                                    (bImportCert > 0) ? true : false);
}

/** @} */
/** @} */
/** @} */
