#!/usr/cogent/bin/gamma

/*
 * This example demonstrates some of the basics of list manipulation.
 * Since Gamma uses the SCADALisp engine, it inherits the rich set of
 * list manipulation functions for which LISP is known.
 *
 * This file is best viewed with a tab width of 4.
 */

/*
 * LISP defines the functions 'car', 'cdr', and 'cons' as the basic
 * list manipulation functions.  The origins of these names have no
 * meaning on today's computers:
 *    CAR	- Contents of the Address Register
 *    CDR	- Contents of the Decrement Register
 *    CONS	- Construct (OK, this makes sense)
 *
 * We can define alternate names for car and cdr here:
 */

first := car;
tail := cdr;

/*
 * Create some lists.
 */

a = list ("My", "dog", "has", "fleas");
b = list (1, 2, 3, 4, 5, 6);
c = list (#a, #b, #c, #d, #e, #f);
d = list (#d, #h, #b, #i, #g, #e, #c);

/*
 * Take the first component of a list.
 */

e = first (a);
princ ("The first component of ", a, " is ", e, "\n");

/*
 * Take the tail of a list.
 */

e = tail (a);
princ ("\nThe tail of ", a, " is ", e, "\n");

/*
 * Concatenate two lists.  Notice that list elements do not need to be the
 * same type.
 */

e = append (a, b);
princ ("\nappending ", b, " to ", a, " gives:\n    ", e, "\n");

/*
 * Walk a list and print each element
 */

princ ("\nThe elements of 'a' are:\n");
with i in a do
{
	princ ("    ", i, "\n");
}

/*
 * Add an element to the beginning of a list
 */

princ ("\nAdding a zero to ", b, "\n");
b = cons (0, b);
princ ("    Gives: ", b, "\n");

/*
 * Take the first element of the tail of the list (the second element)
 * We can use combinations of car and cdr to do this.  However, Gamma
 * predefines a number of car and cdr combinations to make this easier,
 * by inserting multiple 'a's and 'd's between the 'c' and the 'r' in
 * the words car and cdr.
 * For example,   caddr(x)    is the same as    car(cdr(cdr(x)))
 *
 * Gamma defines all one_, two_, and three_letter combinations of
 * car and cdr.
 */

princ ("\na is ", a, "\n");
x = car (cdr (a));
princ ("    car(cdr(a)) is: ", x, "\n");
x = cadr (a);
princ ("    cadr(a) is: ", x, "\n");

/*
 * Some other interesting list functions...
 */

princ ("\n");
princ ("Union of        ", c, " and ", d, " is ", union (c, d), "\n");
princ ("Intersection of ", c, " and ", d, " is ", intersection (c, d), "\n");
princ ("Difference of   ", c, " and ", d, " is ", difference (c, d), "\n");
princ ("Difference of   ", d, " and ", c, " is ", difference (d, c), "\n");
