#!/usr/cogent/bin/phgamma

/*
 * After a window is created in code or loaded from a PhAB definition
 * file, one of the most common thigs to do is to attach callbacks to the
 * window items (widgets).  A callback is a block of code which will be
 * executed whenever a particular Photon event occurs.  Gamma greatly
 * enhances the Photon callback mechanism to allow any Gamma expression
 * to be attached to a callback.  This example demonstrates some simple
 * Photon callbacks.
 * 
 * This file is best viewed with a tab width of 4.
 */

/*
 * Initiate the graphics session with the Photon window manager
 */
init_ipc ("example08", nil);
PtInit(nil);

/*
 * Set up the arc angle
 */

arc_angle := 90;

/*
 * Declare a global name for use with example 12
 */

// qnx_name_attach (0, "example08");
 
/*
 * Load Photon Widget convenience functions
 */
require_lisp("PhotonWidgets");

/*
 * Load support for loading windows created in PhAB
 */
require_lisp("PhabTemplate");

/*
 * Load and convert the window created in PhAB using PhabLoad function.
 * Set first item returned (PtWindow) definition to be the variable
 * named win.
 */
win = car(PhabLoad("wgt/example08.wgtw"));

/*
 * Attach callbacks to Button 1, 2, and 3 ... button 1 responds on an
 * Activate (mouse down then up), button 2 to arm (button down), and
 * button 3 on disarm (usually button up).
 * Each of the callbacks has a simple piece of code which sets the
 * text_string resource of the widget named 'Label'.
 */
PtAttachCallback(Button1, Pt_CB_ACTIVATE, #Label.text_string="Activate");
PtAttachCallback(Button2, Pt_CB_ARM, #Label.text_string="Arm");
PtAttachCallback(Button3, Pt_CB_DISARM, #Label.text_string="Disarm");

/*
 * Attach callbacks to buttons Inc and Dec.  Two callbacks
 * are attached for each button ... one on the Activate event, one
 * for the repeat (hold mouse down) event.  All callbacks call
 * the function value_delta with the first arg being the
 * widget to read/modify and the second arg being the delta
 */
PtAttachCallback(Dec, Pt_CB_ACTIVATE, #value_delta(Value, -1));
PtAttachCallback(Dec, Pt_CB_REPEAT, #value_delta(Value, -1));
PtAttachCallback(Inc, Pt_CB_ACTIVATE, #value_delta(Value, 1));
PtAttachCallback(Inc, Pt_CB_REPEAT, #value_delta(Value, 1));

/*
 * Attach a callback on the SLIDER_MOVE callback off of the
 * slider widget and call function rotate_arc() when activated.
 */
PtAttachCallback(Slider, Pt_CB_SLIDER_MOVE, #rotate_arc());

/*
 * Attach a simple callback on the Arc (pie) which responds to
 * a click by the mouse (Activate) ... simple attached code
 * sets label to 'Click!'
 */
PtAttachCallback(Arc, Pt_CB_ACTIVATE, #ArcLabel.text_string = "Click!");

/*
 * Function attached to callbacks on Increment and Decrement buttons
 */
function value_delta (widget, delta)
{
	local num;

	num = number(widget.text_string);
	widget.text_string = string(num + delta);
}

/*
 * Function attached to callback on Slider move.  We rotate the arc by
 * modifying its arc start and end values, in units of 1/10th of a degree.
 */
function rotate_arc ()
{
	Arc.arc_start = Slider.gauge_value * 10;
	Arc.arc_end = (Slider.gauge_value - arc_angle) * 10;
	ArcLabel.text_string = string(Slider.gauge_value);
}

/*
 * Start an infinite event loop to handle Photon events.  We could also
 * use a call to PtMainLoop() here, but then we could not intervene after
 * each event, which we might want to do for some reason.
 */
while (t)
{
	next_event();
}

