/*--------------------------------------------------------------------
 * File:        photon.g
 *
 * Description: Functions that run only in Photon.
 *
 * Methods:
 *              PtLabel.switched_on
 *              PtLabel.get_text
 *              PtLabel.set_text
 *              PtBasic.rollover
 *                PtTrend.put_data
 * Functions:
 *              attach_msg
 *              anygui_show_text
 *              anygui_sigchild
 *              anygui_destroyer
 *              anygui_move_window
 *              anygui_makemsg
 *------------------------------------------------------------------*/

require_lisp("PhotonWidgets.lsp");
require_lisp("PhabTemplate.lsp");
PtInit(nil);

/*--------------------------------------------------------------------
 * Method:      PtLabel.switched_on
 * Returns:     t or nil
 * Description: Creates common method to check Gtk and Photon buttons's 
 *              toggled status.  The on/off flag for a PtButton is
 *              inherited from the parent class, PtLabel.
 *------------------------------------------------------------------*/
method PtLabel.switched_on()
{
  if ((.flags & Pt_SET) != 0)
        t;
  else
        nil;
}

/*--------------------------------------------------------------------
 * Method:      PtLabel.get_text
 * Returns:     a string
 * Description: Emulates the GtkEntry.get_text() method, needed for
 *              the PtText widget, which is a child class of PtLabel.
 *              There is no need for an equivalent method in gtk.g.
 *------------------------------------------------------------------*/
method PtLabel.get_text()
{
  .text_string;
}

/*--------------------------------------------------------------------
 * Method:      PtLabel.set_text
 * Returns:     t or nil
 * Description: Emulates the GtkEntry.set_text() method, needed for
 *              the PtText widget, which is a child class of PtLabel.
 *              There is no need for an equivalent method in gtk.g.
 *------------------------------------------------------------------*/
method PtLabel.set_text(str)
{
  .text_string = str;
}

/*--------------------------------------------------------------------
 * Method:      PtBasic.rollover
 * Returns:     t or nil
 * Description: Called by the attach_msg function to provide rollover
 *              functionality on any widget.  Used to send messages to
 *              the Controller when the pointer is on a button or other
 *              widget.
 *------------------------------------------------------------------*/
method PtBasic.rollover(msg_in, msg_out)
{
  if (cbinfo.event.subtype == Ph_EV_PTR_ENTER)
    send_message(msg_in);
  else if (cbinfo.event.subtype == Ph_EV_PTR_LEAVE)
    send_message(msg_out);
}

/*--------------------------------------------------------------------
 * Function:    PtTrend.put_data
 * Application: Monitor
 * Returns:     t or nil
 * Description: Standardizes the .rttrend_data() for RtTrend in QNX 4
 *              and .trend_data() for PtTrend in QNX 6 into one, common
 *              method call.
 *------------------------------------------------------------------*/
method PtTrend.put_data(array)
{
  .trend_data = array;
}

/*--------------------------------------------------------------------
 * Function:    attach_msg
 * Returns:     t or nil
 * Description: Attaches a PtBasic.rollover demo library method to a widget
 *              boundary event.  Uses Pt_CB_RAW to access the Ph_EV_BOUNDARY
 *              event, since there is no callback associated with a
 *              Ph_EV_BOUNDARY event for a PtBasic widget.
 *------------------------------------------------------------------*/
function attach_msg(widget, msg_in, msg_out)
{
  PtAttachCallback (widget, Pt_CB_RAW,
                    `(@widget).rollover(@msg_in, @msg_out),
                    Ph_EV_BOUNDARY);
}

/*--------------------------------------------------------------------
 * Function:    anygui_show_text
 * Returns:     t or nil
 * Description: Displays a message string in a text widget.  The fnt
 *              argument is only used in the GTK version of this
 *              function, and is ignored by the Photon version.
 *------------------------------------------------------------------*/
function anygui_show_text(wgt, str, fnt)
{
  wgt.text_string = str;
}

/*--------------------------------------------------------------------
 * Function:    anygui_sigchild
 * Returns:     t or nil
 * Description: Pops out a button when the corresponding child dies.
 *------------------------------------------------------------------*/
function anygui_sigchild ()
{
  child_died(#button.flags = cons(Pt_SET, nil));
}

/*--------------------------------------------------------------------
 * Function:    anygui_destroyer
 * Returns:     nil
 * Description: A space-holder for a function only used in GTK.
 *------------------------------------------------------------------*/
function anygui_destroyer(wgt)
{
  nil;
}

/*--------------------------------------------------------------------
 * Function:    anygui_move_window
 * Returns:     nil
 * Description: A space-holder for a function only used in Linux/GTK.
 *------------------------------------------------------------------*/
function anygui_move_window(button, window, old_x, old_y, new_x, new_y)
{
  nil;
}


/*--------------------------------------------------------------------
 * Function:    anyos_change_settings
 * Application: Monitor
 * Returns:     function
 * Description: Used by the Monitor to change PID settings, and optionally
 *              unclick the Auto button if it is on.  The PtNumeric widgets
 *              don't execute callbacks after they have been moved
 *              programmatically, so we have to write the point to the
 *              DataHub using write_point() each time a setting changes.
 *------------------------------------------------------------------*/
function anyos_change_settings(auto, sp1, v1, sp2, v2, sp3, v3, sp4, v4,
                               sp5, v5, sp6, v6, msgno, autobutton?)
{
  if (auto == nil)
    {
      if (AUTO_001 != 0)
        {
          autobutton.onoff_state = 0;
          write_point(#AUTO_001, 0);
        }
    }
  else
    {
      autobutton.onoff_state = 1;
      write_point(#AUTO_001, 1);
      sp1.numeric_value = v1;
      write_point(#FREQ_001, v1);
    }
  anyver_change_settings(sp2, v2, sp3, v3, sp4, v4,
                         sp5, v5, sp6, v6);
  
  write_point(#PID1_Kp, v2);
  write_point(#PID1_Ki, v3);
  write_point(#PID1_Kd, v4);
  write_point(#PROP_001, v5);
  write_point(#INT_001, v6);
}

/*--------------------------------------------------------------------
 * Function:    anygui_makemsg
 * Returns:     t or nil
 * Description: Creates and displays a message window.
 *------------------------------------------------------------------*/
function anygui_makemsg(msg)
{
  if ((undefined_p(msgwin)) || (destroyed_p(msgwin)))
    {
      msgwin = new(PtMessage);
      msgwin.msg_text = msg;
      msgwin.msg_button1 = " OK ";
      PtRealizeWidget(msgwin);
    }
  else
    if (instance_p(msgwin))
      PtDestroyWidget(msgwin);
}

princ("GUI is Photon.\n");
