//Copyright, 2002-2003, Astra Network Inc.  All Rights Reserved

//This source code has been published by Astra Network Inc. However, any
//use, reproduction, modification, distribution or transfer of this
//software, or any software which includes or is based upon any of this
//code, is only permitted if expressly authorized by a written license
//agreement from Astra. Contact your Astra representative directly for
//more information.

/*!
 * \file HC_Normal.c
 * Functions relating to the operation of a nodes Normal function.
 */

#include "checkpoint_int.h"

/*!
 * Transition forward to a new node.
 * \param sequence A sequence object.
 * \param node The node to transition to.
 * \param userdata User specific data.
 * \return An error value.
 */
int
HC_Normal (HC_Sequence_t *sequence, long node, void *userdata)
{
  int rc;

  if (sequence)
    {
      if ((rc = HCI_PushNode (sequence, sequence->curr_node)) != HC_OK)
	return (rc);

      sequence->curr_node = node;

      (void)HCI_CallNormal (sequence, sequence->curr_node, userdata);

      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}

/*!
 * Rerun the normal function for the current node.
 * \param sequence A sequence object.
 * \param userdata User specific data.
 * \return An error value.
 */
int
HC_NormalCurrent (HC_Sequence_t *sequence, void *userdata)
{
  if (sequence)
    {
      (void)HCI_CallNormal (sequence, sequence->curr_node, userdata);

      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}

/*!
 * Called from a normal function signifying success.
 * \param sequence Sequence object.
 * \param policy_data Data specific to the policy
 * \return An error code.
 */
int
HC_NormalSuccess (HC_Sequence_t *sequence, long policy_data)
{
  if (sequence)
    {
      /* Simply change the event to indicate success, and make sure the
         proper policy_data is passed to the policy function */

      sequence->curr_event = HC_NORMALSUCCESS;
      sequence->curr_policy_data = policy_data;

      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}

/*!
 * Called from a normal function signifying failure.
 * \param sequence Sequence object.
 * \param policy_data Data specific to the policy
 * \return An error code.
 */
int
HC_NormalFail (HC_Sequence_t *sequence, long policy_data)
{
  if (sequence)
    {
      /* Change the event to indicate failure, and make sure the proper 
         policy_data gets passed to the policy funcion */
      sequence->curr_event = HC_NORMALFAIL;
      sequence->curr_policy_data = policy_data;

      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}

