//Copyright, 2002-2003, Astra Network Inc.  All Rights Reserved

//This source code has been published by Astra Network Inc. However, any
//use, reproduction, modification, distribution or transfer of this
//software, or any software which includes or is based upon any of this
//code, is only permitted if expressly authorized by a written license
//agreement from Astra. Contact your Astra representative directly for
//more information.

/*!
 * \file HC_Rollback.c
 * Collection of functions relating to rolling back nodes in a
 * sequence.
 */

#include "checkpoint_int.h"

/*!
 * Rolls back the current node.
 * \param sequence Sequence object.
 * \return An error code.
 */
int
HC_RollBackCurrent (HC_Sequence_t *sequence)
{
  int rc;
  if (sequence)
    {
      rc = HCI_CallRollback (sequence, sequence->curr_node,
			     sequence->userdata,
			     sequence->checkpoint);
      if (rc != HC_OK)
	(void) HC_Log (sequence, "Rollback returned failure, %s",
		       HC_Strerror (rc));
      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}

/*!
 * Rolls back the previous node.
 * \param sequence Sequence object.
 * \return An error code.
 */
int
HC_RollBackPrev (HC_Sequence_t *sequence)
{
  int rc;
  if (sequence)
    {
      sequence->curr_node = HCI_PopNode (sequence);

      rc = HCI_CallRollback (sequence, sequence->curr_node,
			     sequence->userdata,
			     sequence->checkpoint);
      if (rc != HC_OK)
	(void) HC_Log (sequence, "Rollback returned failure, %s",
		       HC_Strerror (rc));
      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}

/*!
 * Called from a rollback function signifying success in rolling back.
 * \param sequence Sequence object.
 * \param policy_data Data specific to the policy
 * \return An error code.
 */
int
HC_RollBackSuccess (HC_Sequence_t *sequence, long policy_data)
{
  if (sequence)
    {
      /* Simply change the event to indicate success, and make sure the
         proper policy_data is passed to the policy function */

      sequence->curr_event = HC_ROLLBACKSUCCESS;
      sequence->curr_policy_data = policy_data;

      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}

/*!
 * Called from a rollback function signifying failure in rolling back.
 * \param sequence Sequence object.
 * \param policy_data Data specific to the policy
 * \return An error code.
 */
int
HC_RollBackFail (HC_Sequence_t *sequence, long policy_data)
{
  if (sequence)
    {
      /* Change the event to indicate failure, and make sure the proper 
         policy_data gets passed to the policy funcion */
      sequence->curr_event = HC_ROLLBACKFAIL;
      sequence->curr_policy_data = policy_data;

      return (HC_OK);
    }
  return (HC_INVALIDSEQUENCE);
}
