/*
 * h263codec.h
 *
 * H.323 protocol handler
 *
 * Open H323 Library
 *
 * Copyright (c) 1999-2000 Equivalence Pty. Ltd.
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open H323 Library.
 *
 * The Initial Developer of the Original Code is Equivalence Pty. Ltd.
 *
 * Contributor(s): Guilhem Tardy (gtardy@marchnetworks.com)
 *
 * $Log: h263codec.h,v $
 * Revision 1.2  2002/11/20 04:26:32  xtang
 * Orignal 1-10-4 package.
 *
 * Revision 1.4  2002/09/16 01:14:15  robertj
 * Added #define so can select if #pragma interface/implementation is used on
 *   platform basis (eg MacOS) rather than compiler, thanks Robert Monaghan.
 *
 * Revision 1.3  2002/09/03 06:19:36  robertj
 * Normalised the multi-include header prevention ifdef/define symbol.
 *
 * Revision 1.2  2002/08/05 10:03:47  robertj
 * Cosmetic changes to normalise the usage of pragma interface/implementation.
 *
 * Revision 1.1  2002/05/19 22:32:46  dereks
 * Initial release of stub file for h263 codec. Thanks Guilhem Tardy.
 *
 *
 *
 */
 
/*
 * Initial release notes from Guilhem Tardy::
 *
 * Added support for video capabilities & codec, only tested under Linux!
 * The code for varying bit rate is copied from h261codec.cxx,
 * until it is moved to a separate file common to both video codecs.
 *
 */

#ifndef __OPAL_H263CODEC_H
#define __OPAL_H263CODEC_H

#ifdef P_USE_PRAGMA
#pragma interface
#endif


#include <endian.h>
#include "h323caps.h"


class H263Decoder;
class MSBBitStream;
class H263Encoder;
class H263Parser;



///////////////////////////////////////////////////////////////////////////////

/**This class is a H.263 video capability.
 */
class H323_H263Capability : public H323VideoCapability
{
  PCLASSINFO(H323_H263Capability, H323VideoCapability)

  public:
  /**@name Construction */
  //@{
    /**Create a new H263 Capability
     */
    H323_H263Capability(
      unsigned sqcifMPI,
      unsigned qcifMPI,
      unsigned cifMPI,
      unsigned cif4MPI,
      unsigned cif16MPI,
      unsigned maxBitRate = 850,
      BOOL unrestrictedVector = FALSE,
      BOOL arithmeticCoding = FALSE,
      BOOL advancedPrediction = FALSE,
      BOOL pbFrames = FALSE,
      BOOL temporalSpatialTradeOff = TRUE,
      unsigned hrd_B = 0,
      unsigned bppMaxKb = 0,
      unsigned slowSqcifMPI = 0,
      unsigned slowQcifMPI = 0,
      unsigned slowCifMPI = 0,
      unsigned slowCif4MPI = 0,
      unsigned slowCif16MPI = 0,
      BOOL errorCompensation = FALSE
    );
  //@}

  /**@name Overrides from class PObject */
  //@{
    /**Create a copy of the object.
      */
    virtual PObject * Clone() const;
  //@}

  
  /**@name Overrides from class PObject */
  //@{
    /**Compare object
      */
   Comparison Compare(const PObject & obj) const;
   //@}

  /**@name Identification functions */
  //@{
    /**Get the sub-type of the capability. This is a code dependent on the
       main type of the capability.

       This returns one of the four possible combinations of mode and speed
       using the enum values of the protocol ASN H245_AudioCapability class.
     */
    virtual unsigned GetSubType() const;

    /**Get the name of the media data format this class represents.
     */
    virtual PString GetFormatName() const;
  //@}

  /**@name Protocol manipulation */
  //@{
    /**This function is called whenever and outgoing TerminalCapabilitySet
       or OpenLogicalChannel PDU is being constructed for the control channel.
       It allows the capability to set the PDU fields from information in
       members specific to the class.

       The default behaviour sets the data rate field in the PDU.
     */
    virtual BOOL OnSendingPDU(
      H245_VideoCapability & pdu  /// PDU to set information on
    ) const;

    /**This function is called whenever and outgoing RequestMode
       PDU is being constructed for the control channel. It allows the
       capability to set the PDU fields from information in members specific
       to the class.

       The default behaviour sets the resolution and bit rate.
     */
    virtual BOOL OnSendingPDU(
      H245_VideoMode & pdu  /// PDU to set information on
    ) const;

    /**This function is called whenever and incoming TerminalCapabilitySet
       or OpenLogicalChannel PDU has been used to construct the control
       channel. It allows the capability to set from the PDU fields,
       information in members specific to the class.

       The default behaviour gets the data rate field from the PDU.
     */
    virtual BOOL OnReceivedPDU(
      const H245_VideoCapability & pdu  /// PDU to set information on
    );

    /**Create the codec instance, allocating resources as required.
     */
    virtual H323Codec * CreateCodec(
      H323Codec::Direction direction  /// Direction in which this instance runs
    ) const;

    /** Get sqcifMPI
     */
    unsigned GetSQCIFMPI() const
      { return (sqcifMPI>0?sqcifMPI:0); }

    /** Get qcifMPI
     */
    unsigned GetQCIFMPI() const
      { return (qcifMPI>0?qcifMPI:0); }

    /** Get cifMPI
     */
    unsigned GetCIFMPI() const
      { return (cifMPI>0?cifMPI:0); }

    /** Get cif4MPI
     */
    unsigned GetCIF4MPI() const
      { return (cif4MPI>0?cif4MPI:0); }

    /** Get cif16MPI
     */
    unsigned GetCIF16MPI() const
      { return (cif16MPI>0?cif16MPI:0); }

    /** Get maximum bit rate
     */
    unsigned GetMaxBitRate() const
      { return maxBitRate; }

    /** Get unrestrictedVector capabilty
     */
    BOOL GetUnrestrictedVectorCapability() const
      { return unrestrictedVector; }

    /** Get arithmeticCoding capabilty
     */
    BOOL GetArithmeticCodingCapability() const
      { return arithmeticCoding; }

    /** Get advancedPrediction capabilty
     */
    BOOL GetAdvancedPredictionCapability() const
      { return advancedPrediction; }

    /** Get  pbFrames capabilty
     */
    BOOL GetPbFramesCapability() const
      { return pbFrames; }

    /** Get temporal/spatial tradeoff capabilty
     */
    BOOL GetTemporalSpatialTradeOffCapability() const
      { return temporalSpatialTradeOff; }

    /** Get hrd_B
     */
    BOOL GetHrd_B() const
      { return hrd_B; }

    /** Get bppMaxKb
     */
    BOOL GetBppMaxKb() const
      { return bppMaxKb; }

    /** Get slowSqcifMPI
     */
    unsigned GetSlowSQCIFMPI() const
      { return (sqcifMPI<0?-sqcifMPI:0); }

    /** Get slowQcifMPI
     */
    unsigned GetSlowQCIFMPI() const
      { return (qcifMPI<0?-qcifMPI:0); }

    /** Get slowCifMPI
     */
    unsigned GetSlowCIFMPI() const
      { return (cifMPI<0?-cifMPI:0); }

    /** Get slowCif4MPI
     */
    unsigned GetSlowCIF4MPI() const
      { return (cif4MPI<0?-cif4MPI:0); }

    /** Get slowCif16MPI
     */
    unsigned GetSlowCIF16MPI() const
      { return (cif16MPI<0?-cif16MPI:0); }

    /** Get errorCompensation capabilty
     */
    BOOL GetErrorCompensationCapability() const
      { return errorCompensation; }

  //@}

  protected:
    // H.263 v1
    signed sqcifMPI;		// {1..3600 units seconds/frame, 1..32 units 1/29.97 Hz}
    signed qcifMPI;		// {1..3600 units seconds/frame, 1..32 units 1/29.97 Hz}
    signed cifMPI;		// {1..3600 units seconds/frame, 1..32 units 1/29.97 Hz}
    signed cif4MPI;		// {1..3600 units seconds/frame, 1..32 units 1/29.97 Hz}
    signed cif16MPI;		// {1..3600 units seconds/frame, 1..32 units 1/29.97 Hz}

    unsigned maxBitRate;	// units of 100 bit/s

    BOOL     unrestrictedVector;
    BOOL     arithmeticCoding;
    BOOL     advancedPrediction;
    BOOL     pbFrames;
    BOOL     temporalSpatialTradeOff;

    long unsigned hrd_B;	// units of 128 bits
    unsigned bppMaxKb;		// units of 1024 bits

    BOOL     errorCompensation;
};


/**This class is a H.263 video codec.
 */
class H323_H263Codec : public H323VideoCodec
{
  PCLASSINFO(H323_H263Codec, H323VideoCodec)

  public:
    /**Create a new H263 video codec
     */
    H323_H263Codec(
      Direction direction	/// Direction in which this instance runs
    );

    ~H323_H263Codec();

    /**Encode the data from the appropriate device.
       This will encode a frame of data for transmission. The exact size and
       description of the data placed in the buffer is codec dependent but
       should be less than H323Capability::GetTxFramesInPacket() *
       OpalMediaFormat::GetFrameSize()  in length.

       The length parameter is filled with the actual length of the encoded
       data, often this will be the same as the size parameter.

       This function is called every GetFrameRate() timestamp units, so MUST
       take less than (or equal to) that amount of time to complete!

       Note that a returned length of zero indicates that time has passed but
       there is no data encoded. This is typically used for silence detection
       in an audio codec.
     */
    virtual BOOL Read(
      BYTE * buffer,		/// Buffer of encoded data
      unsigned & length,	/// Actual length of encoded data buffer
      RTP_DataFrame & rtpFrame	/// RTP data frame
    );

    /**Decode the data and output it to appropriate device.
       This will decode a single frame of received data. The exact size and
       description of the data required in the buffer is codec dependent but
       should be less than H323Capability::GetRxFramesInPacket() *
       OpalMediaFormat::GetFrameSize()  in length.

       It is expected this function anunciates the data. That is, for example
       with audio data, the sound is output on a speaker.

       This function is called every GetFrameRate() timestamp units, so MUST
       take less than that amount of time to complete!
     */
    virtual BOOL Write(
      const BYTE * buffer,	  /// Buffer of encoded data
      unsigned length,		  /// Length of encoded data buffer
      const RTP_DataFrame & rtp,  /// RTP data frame
      unsigned & written	  /// Number of bytes used from data buffer
    );

    /**Get the frame rate in RTP timestamp units.
      */
    virtual unsigned GetFrameRate() const { return timestampDelta; }

    /**Set the quality level of transmitted video data. 
       Is irrelevant when this codec is used to receive video data.
       Has a value of 1 (good quality) to 31 (poor quality).
       Quality is improved at the expense of bit rate.
    */
    void SetTxQualityLevel(int qLevel);
 
    /**Set the number of blocks in the background that need
       to be transmitted with each frame
    */
    void SetBackgroundFill(int fillLevel);

    /**Process a request for a new frame, 
       as part of the picture has been lost.
    */
    virtual void OnLostPartialPicture();

    /**Process a request for a new frame, 
       as all of the picture has been lost.
    */
    virtual void OnLostPicture();

    ////////////////////////////////////////////////////////////////
    //There is a problem with the H263codec. It needs to be able to 
    //carry out two tasks. 1)Grab data from the camera.
    //2)Render data from an array.
    //Thus, we either: two PVideoChannels, or one PVideoChannel to both
    //grab and render.
    //We use one PVideoChannel, which is not consistant with elsewhere,
    //but enables us to (later) have a grab and display process irrespective
    //of there being a H323 connection.


  protected:
    BOOL Resize(int width, int height);
    BOOL RenderFrame(const void * buffer);

    /* decoder (manages its own buffer for the raw frame after decoding) */
    H263Decoder		* videoDecoder;

    /* raw frame before encoding */
    unsigned rawFrameBufLen;
    unsigned char	* rawFrameBuffer;
    /* H.263 frame after encoding */
    unsigned encFrameBufLen;
    unsigned char	* encFrameBuffer;
    /* encoder and parser */
    MSBBitStream	* videoEncoderBitStream;
    H263Encoder		* videoEncoder;
    H263Parser		* videoParser;

    unsigned timestampDelta;

    struct rfc2190hdr {
	union {
	    struct {
#if __BYTE_ORDER == __BIG_ENDIAN
		u_int32_t	f:1,	// 0: mode A, 1: mode B or C
				p:1,	// 0: normal I or P (mode B), 1: PB-frame (mode C)
				sbit:3,	// start bit
				ebit:3,	// end bit
				src:3,	// source format
				i:1,	// 0: intra, 1: inter
				u:1,	// UMV
				s:1,	// syntax-based Arithm. Coding
				a:1,	// avd. prediction
				_r:4,	// reserved
				trb:3,	// temporal reference (B frame)
				dbq:2,	// differential quantizer
				tr:8;	// temporal reference (P frame)
#elif __BYTE_ORDER == __LITTLE_ENDIAN
		u_int32_t	tr:8,	// temporal reference (P frame)
				dbq:2,	// differential quantizer
				trb:3,	// temporal reference (B frame)
				_r:4,	// reserved
				a:1,	// avd. prediction
				s:1,	// syntax-based Arithm. Coding
				u:1,	// UMV
				i:1,	// 0: intra, 1: inter
				src:3,	// source format
				ebit:3,	// end bit
				sbit:3,	// start bit
				p:1,	// 0: normal I or P (mode B), 1: PB-frame (mode C)
				f:1;	// 0: mode A, 1: mode B or C
#else
#error	"Please fix <asm/byteorder.h>"
#endif
	    } modeA;
	    struct {
#if __BYTE_ORDER == __BIG_ENDIAN
		u_int32_t	f:1,	// 0: mode A, 1: mode B or C
				p:1,	// 0: normal I or P (mode B), 1: PB-frame (mode C)
				sbit:3,	// start bit
				ebit:3,	// end bit
				src:3,	// source format
				q:5,	// quantizer
				gobn:5,	// GOB number
				mba:9,	// first MB address
				_r1:2;	// reserved
		u_int32_t	i:1,	// 0: intra, 1: inter
				u:1,	// UMV
				s:1,	// syntax-based Arithm. Coding
				a:1,	// adv. prediction
				hmv1:7,	// horizontal motion vector for 1st block
				vmv1:7,	// vertical motion vector for 1st block
				hmv2:7,	// horizontal motion vector for 3rd block (adv. prediction)
				vmv2:7;	// vertical motion vector for 3rd block (adv. prediction)
		u_int32_t	_r2:19,	// reserved
				dbq:2,	// differential quantizer
				trb:3,	// temporal reference (B frame)
				tr:8;	// temporal reference (P frame)
#elif __BYTE_ORDER == __LITTLE_ENDIAN
		u_int32_t	_r1:2,	// reserved
				mba:9,	// first MB address
				gobn:5,	// GOB number
				q:5,	// quantizer
				src:3,	// source format
				ebit:3,	// end bit
				sbit:3,	// start bit
				p:1,	// 0: normal I or P (mode B), 1: PB-frame (mode C)
				f:1;	// 0: mode A, 1: mode B or C
		u_int32_t	vmv2:7,	// vertical motion vector for 3rd block (adv. prediction)
				hmv2:7,	// horizontal motion vector for 3rd block (adv. prediction)
				vmv1:7,	// vertical motion vector for 1st block
				hmv1:7,	// horizontal motion vector for 1st block
				a:1,	// adv. prediction
				s:1,	// syntax-based Arithm. Coding
				u:1,	// UMV
				i:1;	// 0: intra, 1: inter
		u_int32_t	tr:8,	// temporal reference (P frame)
				trb:3,	// temporal reference (B frame)
				dbq:2,	// differential quantizer
				_r2:19;	// reserved
#else
#error	"Please fix <asm/byteorder.h>"
#endif
	    } modeBC;
	}; // union {
    }; // struct rfc2190hdr {

/*
    struct rfc2429hdr {
	u_int16_t	_r:5,		// reserved
			p:1,		// picture/GOB/slice start
			v:1,		// Video Redundancy Coding (VRC) present
			plen:6,		// length of extra picture header (if any)
			pebit:3;	// picture header end bit
	u_int8_t	tid:3,		// Thread ID
			trun:4,		// Thread run
			s:1;		// sync frame
    }; // struct rfc2429hdr {
*/

};


#endif // __OPAL_H263CODEC_H


/////////////////////////////////////////////////////////////////////////////
